<?php
/*******************************************************
 * melder.php – Übersichtliche Verwaltung Meldeempfänger
 * - Einheiten-Scope aus Session
 * - Suche + Filter (Zubehör, Zuordnung, Zeitraum)
 * - CSV-Export
 * - Neu anlegen (Modal)
 * - Aktionen: Bearbeiten, Übergabe-Dokument (Druck)
 *******************************************************/
session_start();
if (!isset($_SESSION['user_id'])) { header('Location: login.php'); exit; }

require_once __DIR__.'/includes/header.php';
require_once __DIR__.'/includes/nav.php';
require_once __DIR__.'/includes/db.php';

function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

$einheit = $_SESSION['einheit'] ?? '';
if ($einheit === '') {
  echo "<div class='container my-5'><div class='alert alert-danger'>❌ Keine Einheit in der Session gefunden.</div></div>";
  require_once __DIR__.'/includes/footer.php';
  exit;
}

// ---------- Flash ----------
if (!isset($_SESSION['flash'])) $_SESSION['flash'] = [];
function flash($t,$m){ $_SESSION['flash'][]=['t'=>$t,'m'=>$m]; }
function flash_render(){
  if (empty($_SESSION['flash'])) return;
  echo "<div class='container mt-3'>";
  foreach($_SESSION['flash'] as $f){
    $map=['success'=>'success','error'=>'danger','warn'=>'warning','info'=>'info'];
    $cls=$map[$f['t']]??'secondary';
    echo "<div class='alert alert-$cls d-flex align-items-center gap-2 mb-2'>{$f['m']}</div>";
  }
  echo "</div>";
  $_SESSION['flash']=[];
}

// ---------- POST: Neu anlegen ----------
if ($_SERVER['REQUEST_METHOD']==='POST' && isset($_POST['melder_hinzufuegen'])) {
  $seriennummer = trim($_POST['seriennummer'] ?? '');
  $typ          = trim($_POST['typ'] ?? '');
  $kennung      = trim($_POST['kennung'] ?? '');
  $ausgabe_datum= !empty($_POST['ausgabe_datum'])? $_POST['ausgabe_datum'] : NULL;
  $ladegeraet   = isset($_POST['ladegeraet']) ? 1 : 0;
  $antenne      = isset($_POST['antenne']) ? 1 : 0;
  $schutztasche = isset($_POST['schutztasche']) ? 1 : 0;
  $kamerad_id   = !empty($_POST['kamerad_id']) ? (int)$_POST['kamerad_id'] : NULL;

  // Kennung nur in eigener Einheit eindeutig
  $stmt = $conn->prepare("SELECT id FROM melder WHERE kennung=? AND einheit=? LIMIT 1");
  $stmt->bind_param('ss', $kennung, $einheit);
  $stmt->execute(); $stmt->store_result();

  if ($stmt->num_rows>0) {
    flash('error', "Fehler: Die Kennung <strong>".h($kennung)."</strong> existiert in deiner Einheit bereits.");
  } elseif ($seriennummer==='' || $typ==='') {
    flash('error', 'Bitte Seriennummer und Typ angeben!');
  } else {
    $ins = $conn->prepare("
      INSERT INTO melder (seriennummer, typ, kennung, ausgabe_datum, ladegeraet, antenne, schutztasche, kamerad_id, einheit)
      VALUES (?,?,?,?,?,?,?,?,?)
    ");
    $ins->bind_param('ssssiiiis', $seriennummer, $typ, $kennung, $ausgabe_datum, $ladegeraet, $antenne, $schutztasche, $kamerad_id, $einheit);
    $ok = $ins->execute();
    $ok ? flash('success','✅ Melder erfolgreich hinzugefügt.')
        : flash('error','❌ Konnte nicht gespeichert werden.');
  }
  header('Location: '.$_SERVER['PHP_SELF']); exit;
}

// ---------- CSV Export ----------
if (isset($_GET['export']) && $_GET['export']==='csv') {
  $filename = 'melder_export_' . date('Y-m-d') . '.csv';
  header('Content-Type: text/csv; charset=utf-8');
  header("Content-Disposition: attachment; filename=\"$filename\"");
  $out = fopen('php://output','w');
  fputcsv($out, ['Kennung','Typ','Seriennummer','Kamerad','Ausgabe-Datum','Zubehör','Einheit']);

  $sql = "
    SELECT m.*, k.vorname, k.nachname
      FROM melder m
 LEFT JOIN kameraden k ON k.id = m.kamerad_id AND k.einheit = ?
     WHERE m.einheit = ?
  ORDER BY m.kennung ASC
  ";
  $st = $conn->prepare($sql);
  $st->bind_param('ss', $einheit, $einheit);
  $st->execute();
  $res = $st->get_result();
  while($r = $res->fetch_assoc()){
    $kamerad = ($r['vorname']??'') ? ($r['vorname'].' '.$r['nachname']) : '';
    $z = trim(
      ($r['ladegeraet']?'Ladegerät ':'').
      ($r['antenne']?'Antenne ':'').
      ($r['schutztasche']?'Schutztasche':'')
    );
    fputcsv($out, [
      $r['kennung'],
      $r['typ'],
      $r['seriennummer'],
      $kamerad,
      $r['ausgabe_datum']? date('d.m.Y', strtotime($r['ausgabe_datum'])) : '',
      $z,
      $r['einheit']
    ]);
  }
  fclose($out); exit;
}

// ---------- Filter / Suche ----------
$q         = trim($_GET['q'] ?? '');            // Volltext
$hatZub    = trim($_GET['z'] ?? '');            // ladegeraet|antenne|schutztasche
$zugeordnet= trim($_GET['zu'] ?? '');           // yes|no
$von       = trim($_GET['von'] ?? '');          // Ausgabedatum von
$bis       = trim($_GET['bis'] ?? '');          // Ausgabedatum bis

$where  = "m.einheit = ?";
$types  = "s";
$params = [$einheit];

if ($q!==''){
  $where .= " AND (m.seriennummer LIKE ? OR m.typ LIKE ? OR m.kennung LIKE ?)";
  $like = "%$q%";
  $params[]=$like; $types.='s';
  $params[]=$like; $types.='s';
  $params[]=$like; $types.='s';
}

if (in_array($hatZub, ['ladegeraet','antenne','schutztasche'], true)) {
  $where .= " AND m.$hatZub = 1";
}

if ($zugeordnet==='yes') { $where .= " AND m.kamerad_id IS NOT NULL"; }
if ($zugeordnet==='no')  { $where .= " AND m.kamerad_id IS NULL"; }

if ($von!=='' && preg_match('/^\d{4}-\d{2}-\d{2}$/',$von)) {
  $where .= " AND m.ausgabe_datum >= ?"; $params[]=$von; $types.='s';
}
if ($bis!=='' && preg_match('/^\d{4}-\d{2}-\d{2}$/',$bis)) {
  $where .= " AND m.ausgabe_datum <= ?"; $params[]=$bis; $types.='s';
}

// ---------- Query ----------
$sql = "
  SELECT m.*, k.vorname, k.nachname
    FROM melder m
LEFT JOIN kameraden k ON k.id = m.kamerad_id AND k.einheit = ?
   WHERE $where
ORDER BY m.kennung ASC, m.id DESC
";
$params2 = array_merge([$einheit], $params);
$types2  = 's'.$types;
$st = $conn->prepare($sql);
$bind = [$types2]; foreach($params2 as $i=>$v){ $bind[] = &$params2[$i]; }
call_user_func_array([$st,'bind_param'],$bind);
$st->execute();
$melder = $st->get_result();

// ---------- Kameraden fürs Modal ----------
$kk = $conn->prepare("SELECT id, vorname, nachname FROM kameraden WHERE einheit=? ORDER BY nachname, vorname");
$kk->bind_param('s',$einheit);
$kk->execute();
$kameraden = $kk->get_result();
?>

<?php flash_render(); ?>

<div class="container my-4">
  <div class="d-flex flex-wrap justify-content-between align-items-center gap-3">
    <div>
      <h2 class="mb-0">📟 Meldeempfänger</h2>
      <div class="text-muted small">Einheit: <span class="badge text-bg-secondary"><?= h($einheit) ?></span></div>
    </div>
    <div class="d-flex gap-2">
      <a href="?export=csv" class="btn btn-outline-secondary">
        <i class="bi bi-download"></i> CSV Export
      </a>
      <button class="btn btn-success" data-bs-toggle="modal" data-bs-target="#melderModal">
        <i class="bi bi-plus-circle"></i> Neuer Melder
      </button>
    </div>
  </div>

  <!-- Filter / Suche -->
  <div class="card shadow-sm mt-3">
    <div class="card-body">
      <form method="get" class="row g-2 align-items-end">
        <div class="col-md-4">
          <label class="form-label">Suche</label>
          <input type="text" name="q" class="form-control" placeholder="Kennung, Typ, Seriennummer" value="<?= h($q) ?>">
        </div>
        <div class="col-md-2">
          <label class="form-label">Zubehör</label>
          <select name="z" class="form-select">
            <option value="">alle</option>
            <option value="ladegeraet"   <?= $hatZub==='ladegeraet'?'selected':'' ?>>Ladegerät</option>
            <option value="antenne"      <?= $hatZub==='antenne'?'selected':'' ?>>Antenne</option>
            <option value="schutztasche" <?= $hatZub==='schutztasche'?'selected':'' ?>>Schutztasche</option>
          </select>
        </div>
        <div class="col-md-2">
          <label class="form-label">Zuordnung</label>
          <select name="zu" class="form-select">
            <option value="">alle</option>
            <option value="yes" <?= $zugeordnet==='yes'?'selected':'' ?>>zugeordnet</option>
            <option value="no"  <?= $zugeordnet==='no'?'selected':''  ?>>frei</option>
          </select>
        </div>
        <div class="col-md-2">
          <label class="form-label">Von</label>
          <input type="date" name="von" class="form-control" value="<?= h($von) ?>">
        </div>
        <div class="col-md-2">
          <label class="form-label">Bis</label>
          <input type="date" name="bis" class="form-control" value="<?= h($bis) ?>">
        </div>
        <div class="col-12 d-flex gap-2">
          <button class="btn btn-primary"><i class="bi bi-search"></i> Anwenden</button>
          <a class="btn btn-outline-secondary" href="melder.php"><i class="bi bi-x-circle"></i> Zurücksetzen</a>
        </div>
      </form>
    </div>
  </div>

  <!-- Liste -->
  <div class="card shadow-sm mt-3">
    <div class="table-responsive">
      <table class="table table-hover align-middle mb-0">
        <thead class="table-light">
          <tr>
            <th style="min-width:120px;">Kennung</th>
            <th>Typ</th>
            <th>Seriennummer</th>
            <th>Kamerad</th>
            <th style="min-width:120px;">Ausgabe</th>
            <th>Zubehör</th>
            <th class="text-end" style="width:210px;">Aktion</th>
          </tr>
        </thead>
        <tbody>
        <?php if ($melder->num_rows===0): ?>
          <tr><td colspan="7" class="text-center text-muted py-4">Keine Einträge gefunden.</td></tr>
        <?php else: while($m = $melder->fetch_assoc()):
          $kameradName = ($m['vorname']??'') ? trim($m['vorname'].' '.$m['nachname']) : '';
          $badgeZu = $kameradName
            ? "<span class='badge text-bg-success'><i class=\"bi bi-person-check\"></i> zugeordnet</span>"
            : "<span class='badge text-bg-secondary'><i class=\"bi bi-person\"></i> frei</span>";
          $zube = [];
          if ((int)$m['ladegeraet']===1)   $zube[] = "<span class='badge rounded-pill text-bg-dark'>Ladegerät</span>";
          if ((int)$m['antenne']===1)      $zube[] = "<span class='badge rounded-pill text-bg-dark'>Antenne</span>";
          if ((int)$m['schutztasche']===1) $zube[] = "<span class='badge rounded-pill text-bg-dark'>Schutztasche</span>";
          $zhtml = empty($zube)? "<span class='text-muted'>–</span>" : implode(' ', $zube);
          $ausgabe = $m['ausgabe_datum'] ? date('d.m.Y', strtotime($m['ausgabe_datum'])) : '–';
        ?>
          <tr>
            <td><span class="fw-semibold"><?= h($m['kennung']) ?></span></td>
            <td><?= h($m['typ']) ?></td>
            <td><span class="text-monospace"><?= h($m['seriennummer']) ?></span></td>
            <td>
              <?= $kameradName ? h($kameradName) : '<span class="text-muted">–</span>' ?><br>
              <?= $badgeZu ?>
            </td>
            <td><?= $ausgabe ?></td>
            <td><?= $zhtml ?></td>
            <td class="text-end">
              <div class="btn-group">
                <a href="melder_edit.php?id=<?= (int)$m['id'] ?>" class="btn btn-sm btn-outline-primary">
                  <i class="bi bi-pencil-square"></i> Bearbeiten
                </a>
                <!-- Übergabe-Dokument (druckfertig) -->
                <!-- Übergabe-Dokument (druckfertig) -->
                <a href="druck_melder.php?id=<?= (int)$m['id'] ?>" class="btn btn-sm btn-primary" target="_blank" rel="noopener">
                  <i class="bi bi-file-earmark-text"></i> Übergabe-Protokoll
                </a>

              </div>
            </td>
          </tr>
        <?php endwhile; endif; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<!-- Modal: Neuer Melder -->
<div class="modal fade" id="melderModal" tabindex="-1" aria-labelledby="melderModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-lg">
    <div class="modal-content bg-white">
      <div class="modal-header">
        <h5 class="modal-title" id="melderModalLabel">Neuen Melder hinzufügen</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <form method="post">
        <div class="modal-body">
          <div class="row g-3">
            <div class="col-md-4">
              <label class="form-label">Kennung</label>
              <input type="text" name="kennung" class="form-control" placeholder="z. B. MEL-23-017">
            </div>
            <div class="col-md-4">
              <label class="form-label">Seriennummer *</label>
              <input type="text" name="seriennummer" class="form-control" required>
            </div>
            <div class="col-md-4">
              <label class="form-label">Typ *</label>
              <input type="text" name="typ" class="form-control" required>
            </div>

            <div class="col-md-4">
              <label class="form-label">Ausgabe-Datum</label>
              <input type="date" name="ausgabe_datum" class="form-control">
            </div>

            <div class="col-md-4">
              <label class="form-label">Kamerad (optional)</label>
              <select name="kamerad_id" class="form-select">
                <option value="">– keine Auswahl –</option>
                <?php mysqli_data_seek($kameraden,0); while($k = $kameraden->fetch_assoc()): ?>
                  <option value="<?= (int)$k['id'] ?>"><?= h($k['nachname'].', '.$k['vorname']) ?></option>
                <?php endwhile; ?>
              </select>
            </div>

            <div class="col-md-4">
              <label class="form-label">Einheit</label>
              <input type="text" class="form-control" value="<?= h($einheit) ?>" disabled>
              <div class="form-text">Wird automatisch aus der Session gesetzt.</div>
            </div>

            <div class="col-12">
              <label class="form-label d-block">Zubehör</label>
              <div class="d-flex flex-wrap gap-3">
                <div class="form-check">
                  <input class="form-check-input" type="checkbox" name="ladegeraet" id="ladegeraet">
                  <label class="form-check-label" for="ladegeraet">Ladegerät</label>
                </div>
                <div class="form-check">
                  <input class="form-check-input" type="checkbox" name="antenne" id="antenne">
                  <label class="form-check-label" for="antenne">Antenne</label>
                </div>
                <div class="form-check">
                  <input class="form-check-input" type="checkbox" name="schutztasche" id="schutztasche">
                  <label class="form-check-label" for="schutztasche">Schutztasche</label>
                </div>
              </div>
            </div>

          </div> <!-- /row -->
        </div>
        <div class="modal-footer">
          <button type="submit" name="melder_hinzufuegen" class="btn btn-primary">
            <i class="bi bi-save"></i> Speichern
          </button>
          <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Schließen</button>
        </div>
      </form>
    </div>
  </div>
</div>

<?php require_once __DIR__.'/includes/footer.php'; ?>
