<?php
// kamerad_edit.php – Kamerad bearbeiten (Modern UI)
ob_start(); // Start Buffering
session_start();

require_once __DIR__ . '/includes/header.php';
require_once __DIR__ . '/includes/nav.php';
require_once __DIR__ . '/includes/db.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$id = (int)($_GET['id'] ?? 0);
$meldung = '';
$user_einheit = $_SESSION['einheit'] ?? '';

// --- Kamerad laden (nur eigene Einheit) ---
$stmt = $conn->prepare("SELECT * FROM kameraden WHERE id = ? AND einheit = ?");
$stmt->bind_param("is", $id, $user_einheit);
$stmt->execute();
$kamerad = $stmt->get_result()->fetch_assoc();
$stmt->close();

if (!$kamerad) {
    echo "<div class='container my-5'><div class='alert alert-danger'>Kamerad nicht gefunden oder gehört nicht zu deiner Einheit.</div></div>";
    include 'includes/footer.php';
    ob_end_flush();
    exit;
}

// --- Spind-Zuordnungen der EIGENEN Einheit prüfen (Alle außer dem aktuellen Kameraden) ---
$spindnummern = range(1, 80);
$belegte_spinde = [];
$stmt = $conn->prepare("
    SELECT id, spindnr, vorname, nachname
    FROM kameraden
    WHERE id <> ? AND einheit = ? AND spindnr IS NOT NULL AND spindnr <> ''
");
$stmt->bind_param("is", $id, $user_einheit);
$stmt->execute();
$result = $stmt->get_result();
while ($row = $result->fetch_assoc()) {
    $belegte_spinde[$row['spindnr']] = htmlspecialchars($row['vorname'] . ' ' . $row['nachname']);
}
$stmt->close();

// --- Formular speichern ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save'])) {
    $vorname                   = trim($_POST['vorname']);
    $nachname                  = trim($_POST['nachname']);
    $email                     = trim($_POST['email']);
    $spindnr                   = trim($_POST['spindnr']);
    $groesse_jacke             = trim($_POST['groesse_jacke']);
    $groesse_hose              = trim($_POST['groesse_hose']);
    $groesse_hose_agt          = trim($_POST['groesse_hose_agt']);
    $groesse_stiefel           = trim($_POST['groesse_stiefel']);
    $groesse_handschuhe        = trim($_POST['groesse_handschuhe']);
    $groesse_handschuhe_agt    = trim($_POST['groesse_handschuhe_agt']);

    // Spindprüfung
    if ($spindnr !== '' && isset($belegte_spinde[$spindnr])) {
        $meldung = "<div class='alert alert-danger alert-dismissible fade show'>❌ Spind <strong>$spindnr</strong> ist bereits vergeben an {$belegte_spinde[$spindnr]}.<button type='button' class='btn-close' data-bs-dismiss='alert'></button></div>";
    } else {
        // Update
        $stmt = $conn->prepare("
            UPDATE kameraden
               SET vorname = ?, nachname = ?, email = ?, spindnr = ?,
                   groesse_jacke = ?, groesse_hose = ?, groesse_hose_agt = ?,
                   groesse_stiefel = ?, groesse_handschuhe = ?, groesse_handschuhe_agt = ?
             WHERE id = ? AND einheit = ?
        ");
        $stmt->bind_param(
            "ssssssssssis",
            $vorname, $nachname, $email, $spindnr,
            $groesse_jacke, $groesse_hose, $groesse_hose_agt,
            $groesse_stiefel, $groesse_handschuhe, $groesse_handschuhe_agt,
            $id, $user_einheit
        );
        $stmt->execute();
        $affected = $stmt->affected_rows;
        $stmt->close();

        if ($affected === 0) {
            $meldung = "<div class='alert alert-warning alert-dismissible fade show'>⚠️ Es wurden keine Änderungen gespeichert (evtl. keine Änderung oder fehlende Berechtigung).<button type='button' class='btn-close' data-bs-dismiss='alert'></button></div>";
        } else {
            $meldung = "<div class='alert alert-success alert-dismissible fade show'>✅ Kamerad erfolgreich aktualisiert.<button type='button' class='btn-close' data-bs-dismiss='alert'></button></div>";
        }

        // --- RELOAD DATENSATZ ---
        $stmt = $conn->prepare("SELECT * FROM kameraden WHERE id = ? AND einheit = ?");
        $stmt->bind_param("is", $id, $user_einheit);
        $stmt->execute();
        $kamerad = $stmt->get_result()->fetch_assoc();
        $stmt->close();

        // Spindbelegung nach Update neu einlesen
        $belegte_spinde = [];
        $stmt = $conn->prepare("
            SELECT id, spindnr, vorname, nachname
            FROM kameraden
            WHERE id <> ? AND einheit = ? AND spindnr IS NOT NULL AND spindnr <> ''
        ");
        $stmt->bind_param("is", $id, $user_einheit);
        $stmt->execute();
        $result = $stmt->get_result();
        while ($row = $result->fetch_assoc()) {
            $belegte_spinde[$row['spindnr']] = htmlspecialchars($row['vorname'] . ' ' . $row['nachname']);
        }
        $stmt->close();
    }
}
?>

<div class="container my-5">

    <div class="d-flex justify-content-between align-items-center mb-4">
        <h2 class="fw-bold mb-0">
            <i class="bi bi-person-lines-fill text-primary me-2"></i>
            Kamerad bearbeiten
        </h2>
        <a href="kameraden.php" class="btn btn-outline-secondary">
            <i class="bi bi-arrow-left me-1"></i> Zurück zur Liste
        </a>
    </div>

    <?= $meldung ?>

    <form method="post" class="card shadow-sm border-0 p-4">

        <h4 class="text-primary border-bottom pb-2 mb-4">Stammdaten</h4>

        <div class="row g-3 mb-4">
            <div class="col-md-6">
                <label class="form-label small fw-bold">Vorname *</label>
                <div class="input-group">
                    <span class="input-group-text"><i class="bi bi-person"></i></span>
                    <input type="text" name="vorname" value="<?= htmlspecialchars($kamerad['vorname']) ?>" class="form-control" required>
                </div>
            </div>

            <div class="col-md-6">
                <label class="form-label small fw-bold">Nachname *</label>
                <div class="input-group">
                    <span class="input-group-text"><i class="bi bi-person-fill"></i></span>
                    <input type="text" name="nachname" value="<?= htmlspecialchars($kamerad['nachname']) ?>" class="form-control" required>
                </div>
            </div>

            <div class="col-md-6">
                <label class="form-label small fw-bold">E-Mail</label>
                <div class="input-group">
                    <span class="input-group-text"><i class="bi bi-envelope"></i></span>
                    <input type="email" name="email" value="<?= htmlspecialchars($kamerad['email']) ?>" class="form-control">
                </div>
            </div>

            <div class="col-md-6">
                <label class="form-label small fw-bold">Spindnummer *</label>
                <div class="input-group">
                    <span class="input-group-text"><i class="bi bi-key"></i></span>
                    <select name="spindnr" class="form-select font-monospace" required>
                        <option value="" disabled>– Spind wählen –</option>
                        <option value="<?= htmlspecialchars($kamerad['spindnr']) ?>" selected><?= htmlspecialchars($kamerad['spindnr']) ?> (Aktuell)</option>

                        <?php
                        // Nur Spinde anzeigen, die frei sind ODER die aktuell zu diesem Kameraden gehören
                        foreach ($spindnummern as $nr):
                            $ist_eigener = (string)$kamerad['spindnr'] === (string)$nr;
                            $vergeben    = isset($belegte_spinde[$nr]);
                            $text        = $vergeben ? " (vergeben an {$belegte_spinde[$nr]})" : '';

                            // Nur anzeigen, wenn es nicht der eigene und nicht vergeben ist
                            if (!$ist_eigener && !$vergeben):
                        ?>
                            <option value="<?= $nr ?>">
                                <?= $nr ?>
                            </option>
                        <?php
                            endif;
                        endforeach;

                        // Hinweis für vergebene Spinde (die man nicht wählen kann)
                        foreach ($belegte_spinde as $nr => $zuweisung):
                            if ((string)$kamerad['spindnr'] !== (string)$nr):
                        ?>
                            <option value="<?= $nr ?>" disabled style="color:red;">
                                <?= $nr ?> (VERGEBEN an <?= $zuweisung ?>)
                            </option>
                        <?php
                            endif;
                        endforeach;
                        ?>
                    </select>
                </div>
                <div class="form-text">Spindnummern, die bereits belegt sind, sind im Dropdown grau markiert.</div>
            </div>
        </div>

        <h4 class="text-primary border-bottom pb-2 mb-4 mt-4">PSA Größen</h4>

        <div class="row g-3 mb-4">

            <div class="col-md-6">
                <label class="form-label small fw-bold">Einsatzjacke (Größe)</label>
                <input type="text" name="groesse_jacke" value="<?= htmlspecialchars($kamerad['groesse_jacke']) ?>" class="form-control">
            </div>
            <div class="col-md-6">
                <label class="form-label small fw-bold">Einsatzhose (Größe)</label>
                <input type="text" name="groesse_hose" value="<?= htmlspecialchars($kamerad['groesse_hose']) ?>" class="form-control">
            </div>
            <div class="col-md-6">
                <label class="form-label small fw-bold">Einsatzhose AGT (Größe)</label>
                <input type="text" name="groesse_hose_agt" value="<?= htmlspecialchars($kamerad['groesse_hose_agt']) ?>" class="form-control">
            </div>
            <div class="col-md-6">
                <label class="form-label small fw-bold">Stiefel (Größe)</label>
                <input type="text" name="groesse_stiefel" value="<?= htmlspecialchars($kamerad['groesse_stiefel']) ?>" class="form-control">
            </div>
            <div class="col-md-6">
                <label class="form-label small fw-bold">Handschuhe (Größe)</label>
                <input type="text" name="groesse_handschuhe"
                       value="<?= htmlspecialchars($kamerad['groesse_handschuhe']) ?>"
                       class="form-control">
            </div>
            <div class="col-md-6">
                <label class="form-label small fw-bold">Handschuhe AGT (Größe)</label>
                <input type="text" name="groesse_handschuhe_agt"
                       value="<?= htmlspecialchars($kamerad['groesse_handschuhe_agt']) ?>"
                       class="form-control">
            </div>
        </div>

        <div class="d-flex justify-content-end gap-2">
            <a href="kameraden.php" class="btn btn-outline-secondary">
                 <i class="bi bi-x-lg me-1"></i> Abbrechen
            </a>
            <button type="submit" name="save" class="btn btn-primary">
                <i class="bi bi-save me-1"></i> Änderungen speichern
            </button>
        </div>
    </form>
</div>

<?php include 'includes/footer.php'; ob_end_flush(); ?>
