<?php
include 'includes/header.php';
include 'includes/nav.php';
include 'includes/db.php';

session_start();
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

$user_rechte = $_SESSION['is_admin'] ?? 0;
$meldung = '';
$user_einheit = $_SESSION['einheit'] ?? '';

// --- Artikel löschen (nur innerhalb eigener Einheit) ---
if (isset($_GET['delete']) && is_numeric($_GET['delete']) && $user_rechte > 0) {
    $delId = (int)$_GET['delete'];
    $stmt = $conn->prepare("DELETE FROM lager_bestand WHERE id = ? AND einheit = ?");
    $stmt->bind_param("is", $delId, $user_einheit);
    if ($stmt->execute() && $stmt->affected_rows > 0) {
        $meldung = "<div class='alert alert-success'>🗑️ Artikel wurde gelöscht.</div>";
    } else {
        $meldung = "<div class='alert alert-danger'>❌ Fehler beim Löschen oder keine Berechtigung.</div>";
    }
}

// --- Artikelnummer generieren (optional: laufend je Einheit) ---
function generiere_artikelnummer($kategorie, $conn, $einheit) {
    $prefix_map = [
        'PSA-Einsatzjacke' => 'PSA',
        'PSA-Einsatzhose' => 'PSA',
        'PSA-Einsatzhose AGT' => 'PSA',
        'PSA-Handschuhe' => 'PSA',
        'PSA-Helmlampe' => 'PSA',
        'PSA-Helm' => 'PSA',
        'PSA-Stiefel' => 'PSA',
        'Dienstbekleidung' => 'DB',
        'Uniform' => 'UNI',
        'Diverse' => 'DIV'
    ];
    $prefix = $prefix_map[$kategorie] ?? 'ART';

    // Zähle nur innerhalb der eigenen Einheit, um Nummernkonflikte zu vermeiden
    $sql = "SELECT COUNT(*) AS anzahl
            FROM lager_bestand
            WHERE artikelnummer LIKE CONCAT(?, '-%') AND einheit = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ss", $prefix, $einheit);
    $stmt->execute();
    $res = $stmt->get_result()->fetch_assoc();
    $laufnummer = (int)$res['anzahl'] + 1;

    return $prefix . '-' . str_pad($laufnummer, 4, '0', STR_PAD_LEFT);
}

// --- Artikel speichern (immer der eigenen Einheit zuordnen) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['artikel_speichern'])) {
    $artikelnummer = generiere_artikelnummer($_POST['kategorie'], $conn, $user_einheit);

    $stmt = $conn->prepare("INSERT INTO lager_bestand
        (artikelnummer, artikelname, kategorie, groesse, bestand, mindesbestand, einheit)
        VALUES (?, ?, ?, ?, ?, ?, ?)");
    $stmt->bind_param(
        "ssssiss",
        $artikelnummer,
        $_POST['artikelname'],
        $_POST['kategorie'],
        $_POST['groesse'],
        $_POST['bestand'],
        $_POST['mindesbestand'],
        $user_einheit
    );
    if ($stmt->execute()) {
        $meldung = "<div class='alert alert-success'>✅ Artikel wurde erfolgreich gespeichert.</div>";
    } else {
        $meldung = "<div class='alert alert-danger'>❌ Fehler beim Speichern.</div>";
    }
}

// --- Artikel abrufen (immer nur eigene Einheit anzeigen) ---
if ($user_rechte == 2) {
    // z. B. PSA-Viewer
    $stmt = $conn->prepare("SELECT * FROM lager_bestand
                            WHERE kategorie LIKE 'PSA%' AND einheit = ?
                            ORDER BY artikelname ASC");
    $stmt->bind_param("s", $user_einheit);
} elseif ($user_rechte == 3) {
    // z. B. Dienstbekleidung/Uniform-Viewer
    $stmt = $conn->prepare("SELECT * FROM lager_bestand
                            WHERE kategorie IN ('Dienstbekleidung', 'Uniform') AND einheit = ?
                            ORDER BY artikelname ASC");
    $stmt->bind_param("s", $user_einheit);
} else {
    // Standard: alles der eigenen Einheit
    $stmt = $conn->prepare("SELECT * FROM lager_bestand
                            WHERE einheit = ?
                            ORDER BY kategorie, artikelname ASC");
    $stmt->bind_param("s", $user_einheit);
}
$stmt->execute();
$artikel = $stmt->get_result();
?>

<div class="container my-5">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h2>Lagerverwaltung</h2>
    <button class="btn btn-success" data-bs-toggle="modal" data-bs-target="#artikelModal">➕ Artikel hinzufügen</button>
  </div>

  <?= $meldung ?>

  <div class="table-responsive">
    <table class="table table-bordered table-hover bg-white align-middle">
      <thead class="table-dark">
        <tr>
          <th>Artikelnummer</th>
          <th>Name</th>
          <th>Kategorie</th>
          <th>Größe</th>
          <th>Bestand</th>
          <th>Aktion</th>
        </tr>
      </thead>
      <tbody>
        <?php while ($row = $artikel->fetch_assoc()): ?>
          <?php $warnung = ((int)$row['bestand'] <= (int)$row['mindesbestand']); ?>
          <tr class="<?= $warnung ? 'table-danger' : '' ?>">
            <td><?= htmlspecialchars($row['artikelnummer']) ?></td>
            <td><?= htmlspecialchars($row['artikelname']) ?></td>
            <td><?= htmlspecialchars($row['kategorie']) ?></td>
            <td><?= htmlspecialchars($row['groesse']) ?></td>
            <td>
              <?= (int)$row['bestand'] ?>
              <?php if ($warnung): ?>
                <span class="badge bg-danger ms-2">Mindestbestand!</span>
              <?php endif; ?>
            </td>
            <td>
              <a href="lager_buchung.php?id=<?= (int)$row['id'] ?>" class="btn btn-sm btn-outline-primary">Buchen</a>
              <a href="lager_historie.php?id=<?= (int)$row['id'] ?>" class="btn btn-sm btn-outline-secondary">Historie</a>
              <a href="bestellwesen.php?artikel_id=<?= (int)$row['id'] ?>" class="btn btn-sm btn-outline-success" title="Artikel bestellen">🛒</a>
              <?php if ($user_rechte >= 1): ?>
                <a href="?delete=<?= (int)$row['id'] ?>" class="btn btn-sm btn-outline-danger"
                   onclick="return confirm('❗ Diesen Artikel wirklich löschen?');">Löschen</a>
              <?php endif; ?>
            </td>
          </tr>
        <?php endwhile; ?>
      </tbody>
    </table>
  </div>
</div>

<!-- Modal: Artikel hinzufügen -->
<div class="modal fade" id="artikelModal" tabindex="-1" aria-labelledby="artikelModalLabel" aria-hidden="true">
  <div class="modal-dialog">
    <form method="post" class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="artikelModalLabel">Artikel hinzufügen</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <div class="mb-3"><label class="form-label">Artikelname</label><input name="artikelname" class="form-control" required></div>
        <div class="mb-3">
          <label class="form-label">Kategorie</label>
          <select name="kategorie" id="kategorieSelect" class="form-select" required>
            <option value="">– bitte wählen –</option>
            <option>PSA-Einsatzjacke</option>
            <option>PSA-Einsatzhose</option>
            <option>PSA-Einsatzhose AGT</option>
            <option>PSA-Handschuhe</option>
            <option>PSA-Helmlampe</option>
            <option>PSA-Helm</option>
            <option>PSA-Stiefel</option>
            <option>Dienstbekleidung</option>
            <option>Uniform</option>
            <option>Diverse</option>
          </select>
        </div>
        <div class="mb-3"><label class="form-label">Größe</label><input name="groesse" class="form-control"></div>
        <div class="mb-3"><label class="form-label">Bestand</label><input type="number" name="bestand" class="form-control" value="0" min="0"></div>
        <div class="mb-3"><label class="form-label">Mindestbestand</label><input type="number" name="mindesbestand" class="form-control" value="0" min="0"></div>
      </div>
      <div class="modal-footer">
        <button type="submit" name="artikel_speichern" class="btn btn-primary">Speichern</button>
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Abbrechen</button>
      </div>
    </form>
  </div>
</div>

<?php include 'includes/footer.php'; ?>
