<?php
// Inventur – Barcode-Scan/Suche (nur IST aktualisieren; Fahrzeugnamen via ffw_fahrzeuge; lscan=1 nach Speichern)

// --- Includes (stellt $conn oder $mysqli bereit) ---
require_once __DIR__ . '/includes/header.php';
require_once __DIR__ . '/includes/nav.php';
require_once __DIR__ . '/includes/db.php';

// --- DB Helper: nutzt ausschließlich includes/db.php ---
/** @return mysqli */
function db() {
    if (isset($GLOBALS['conn']) && $GLOBALS['conn'] instanceof mysqli) {
        $GLOBALS['conn']->set_charset('utf8mb4');
        return $GLOBALS['conn'];
    }
    if (isset($GLOBALS['mysqli']) && $GLOBALS['mysqli'] instanceof mysqli) {
        $GLOBALS['mysqli']->set_charset('utf8mb4');
        return $GLOBALS['mysqli'];
    }
    die('DB-Verbindung nicht gefunden. Bitte sicherstellen, dass includes/db.php $conn oder $mysqli setzt.');
}
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function toInt($v){ return (int)preg_replace('/\D/','',(string)($v??0)); }

if (session_status() === PHP_SESSION_NONE) session_start();
$einheit = $_SESSION['einheit'] ?? null;
$mysqli  = db();

$info = $error = null;

// ---------- POST: nur IST (anzahlist) speichern + lscan=1 ----------
if ($_SERVER['REQUEST_METHOD']==='POST' && isset($_POST['update_id'])) {
    $id     = (int)$_POST['update_id'];
    $istRaw = trim((string)($_POST['ist'] ?? ''));
    $ist    = preg_match('/^\d+$/', $istRaw) ? $istRaw : '0';

    // SOLL bleibt unverändert; ausgesonderte nicht updaten; lscan auf 1 setzen
    $sql = "UPDATE ffw_geraete
            SET anzahlist = ?, laenderung = ?, lscan = '1'
            WHERE id = ?
              AND LOWER(COALESCE(status,'')) <> 'ausgesondert'";
    if ($einheit) $sql .= " AND einheit = ?";

    if ($stmt = $mysqli->prepare($sql)) {
        $ts = date('Y-m-d H:i:s');
        if ($einheit) $stmt->bind_param('ssis', $ist, $ts, $id, $einheit);
        else          $stmt->bind_param('ssi',  $ist, $ts, $id);
        if ($stmt->execute()) {
            $info = $stmt->affected_rows > 0 ? "IST gespeichert und lscan gesetzt." : "Keine Änderung (ggf. identischer Wert oder Gerät ausgesondert).";
        } else {
            $error = 'Update-Fehler: '.$stmt->error;
        }
        $stmt->close();
    } else {
        $error = 'Fehler beim Vorbereiten des Updates: '.$mysqli->error;
    }
    // Nach Speichern wieder Detailansicht zeigen
    $_GET['id'] = (string)$id;
}

// ---------- GET: Suche/Detail ----------
$q        = trim((string)($_GET['q'] ?? ''));
$detailId = isset($_GET['id']) ? (int)$_GET['id'] : 0;

$detail  = null;
$list    = [];
$headline= '';

/*
 Basisselekt mit JOIN auf ffw_fahrzeuge:
 - holt Fahrzeugnamen sauber aus der Tabelle
 - robuste Verknüpfung:
     g.fahrzeug = f.funkrufname
  OR g.fahrzeug = f.fahrzeugkurz
  OR g.fahrzeug = f.fahrzeugbezeichnung
  OR g.fahrzeug = f.barcode_fzg
*/
$selectBase = "
  SELECT
    g.id, g.barcode, g.bezeichnung, g.gr, g.standort,
    g.fahrzeug, COALESCE(g.status,'') AS status,
    COALESCE(g.anzahl,'0')    AS anzahl,
    COALESCE(g.anzahlist,'0') AS anzahlist,
    f.fahrzeugbezeichnung, f.fahrzeugname, f.fahrzeugkurz, f.funkrufname, f.kennzeichen, f.barcode_fzg
  FROM ffw_geraete g
  LEFT JOIN ffw_fahrzeuge f
    ON  g.fahrzeug = f.funkrufname
    OR  g.fahrzeug = f.fahrzeugkurz
    OR  g.fahrzeug = f.fahrzeugbezeichnung
    OR  g.fahrzeug = f.barcode_fzg
";

if ($detailId) {
    $sql = $selectBase . " WHERE g.id = ?";
    if ($einheit) $sql .= " AND g.einheit = ?";
    if ($stmt = $mysqli->prepare($sql)) {
        if ($einheit) $stmt->bind_param('is', $detailId, $einheit);
        else          $stmt->bind_param('i',  $detailId);
        $stmt->execute();
        $detail = $stmt->get_result()->fetch_assoc();
        $stmt->close();
    }
} elseif ($q !== '') {
    // 1) Exakter Barcode
    $sql = $selectBase . " WHERE g.barcode = ?";
    if ($einheit) $sql .= " AND g.einheit = ?";
    $sql .= " ORDER BY g.id ASC";
    if ($stmt = $mysqli->prepare($sql)) {
        if ($einheit) $stmt->bind_param('ss', $q, $einheit);
        else          $stmt->bind_param('s',  $q);
        $stmt->execute();
        $exact = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt->close();

        if (count($exact) === 1) {
            $detail = $exact[0];
        } elseif (count($exact) > 1) {
            $list = $exact; $headline = 'Mehrere exakte Treffer';
        } else {
            // 2) LIKE-Fallback
            $like = "%".$q."%";
            $sql = $selectBase . " WHERE g.barcode LIKE ?";
            if ($einheit) $sql .= " AND g.einheit = ?";
            $sql .= " ORDER BY g.bezeichnung ASC, g.id ASC LIMIT 50";
            if ($stmt = $mysqli->prepare($sql)) {
                if ($einheit) $stmt->bind_param('ss', $like, $einheit);
                else          $stmt->bind_param('s',  $like);
                $stmt->execute();
                $list = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
                $stmt->close();
                $headline = 'Ähnliche Barcodes';
            }
        }
    }
}

// Hilfsfunktion für Delta-Badge
function deltaClass($d){ return $d===0?'zero':($d>0?'pos':'neg'); }

// Anzeige-Helper für Fahrzeug:
function fahrzeug_label(array $r): string {
    $label = $r['fahrzeugbezeichnung'] ?: ($r['fahrzeugname'] ?: ($r['fahrzeugkurz'] ?: $r['fahrzeug']));
    $parts = [];
    if ($label)                     $parts[] = $label;
    if (!empty($r['funkrufname']))  $parts[] = $r['funkrufname'];
    if (!empty($r['kennzeichen']))  $parts[] = $r['kennzeichen'];
    return implode(' — ', $parts);
}
?>
<style>
  .inv-card{ border-radius:14px; box-shadow:0 8px 24px rgba(2,6,23,.06); }
  .muted{ color:#6b7280; }
  .badge-delta{ font-variant-numeric: tabular-nums; }
  .badge-delta.zero{ background:#e9ecef; color:#495057; }
  .badge-delta.pos{ background:#e6f6ed; color:#107e3e; border:1px solid #c7ebd3; }
  .badge-delta.neg{ background:#fde7e9; color:#b42318; border:1px solid #fac5cb; }
  .qty{ max-width:140px; }
  .alert-aus{ background:#fff1f2; border:1px solid #fecdd3; color:#9f1239; }
</style>

<div class="container my-4">
  <div class="card inv-card">
    <div class="card-body">
      <h1 class="h4 mb-3">Inventur – Barcode</h1>

      <form class="row g-2 mb-3" method="get" action="">
        <div class="col-12 col-md-9">
          <label class="form-label">Barcode scannen oder eingeben</label>
          <input type="text" name="q" value="<?= h($q) ?>" class="form-control" placeholder="z. B. 90110" autofocus>
        </div>
        <div class="col-12 col-md-3 d-flex align-items-end">
          <button class="btn btn-primary w-100" type="submit">Suchen</button>
        </div>
      </form>

      <?php if ($error): ?>
        <div class="alert alert-danger"><?= h($error) ?></div>
      <?php elseif ($info): ?>
        <div class="alert alert-success"><?= h($info) ?></div>
      <?php endif; ?>

      <?php
      // ===== Detailansicht =====
      if ($detail):
        $sol = toInt($detail['anzahl']);
        $ist = toInt($detail['anzahlist']);
        $d   = $ist - $sol;
        $dc  = deltaClass($d);
        $isAus = (mb_strtolower(trim($detail['status']))==='ausgesondert');
      ?>
        <?php if ($isAus): ?>
          <div class="alert alert-aus"><strong>Ausgesondert:</strong> Dieses Gerät ist als „ausgesondert“ markiert und kann nicht bearbeitet werden.</div>
        <?php endif; ?>

        <div class="table-responsive">
          <table class="table align-middle">
            <thead class="table-light">
              <tr>
                <th>Bezeichnung</th>
                <th>Barcode</th>
                <th>GR</th>
                <th>Fahrzeug</th>
                <th>Standort</th>
                <th class="text-end">SOLL</th>
                <th class="text-end">IST</th>
                <th class="text-end">Δ</th>
              </tr>
            </thead>
            <tbody>
              <tr>
                <td><strong><?= h($detail['bezeichnung']) ?></strong></td>
                <td class="text-nowrap"><?= h($detail['barcode']) ?></td>
                <td><?= h($detail['gr']) ?></td>
                <td><?= h(fahrzeug_label($detail)) ?></td>
                <td><?= h($detail['standort']) ?></td>
                <td class="text-end"><strong id="cellSoll"><?= $sol ?></strong></td>
                <td class="text-end">
                  <form method="post" class="d-inline" action="">
                    <input type="hidden" name="update_id" value="<?= (int)$detail['id'] ?>">
                    <div class="d-flex justify-content-end gap-2">
                      <div class="btn-group btn-group-sm" role="group">
                        <button class="btn btn-outline-secondary" type="button" id="btnDec" <?= $isAus?'disabled':'' ?>>−1</button>
                        <button class="btn btn-outline-secondary" type="button" id="btnInc" <?= $isAus?'disabled':'' ?>>+1</button>
                        <button class="btn btn-outline-secondary" type="button" id="btnReset" title="IST = SOLL" <?= $isAus?'disabled':'' ?>>Reset</button>
                      </div>
                      <input type="number" name="ist" id="istInput" class="form-control text-end qty" value="<?= $ist ?>" min="0" step="1" <?= $isAus?'disabled':'' ?>>
                      <button type="submit" class="btn btn-primary" <?= $isAus?'disabled':'' ?>>Speichern</button>
                    </div>
                  </form>
                </td>
                <td class="text-end"><span id="deltaBadge" class="badge badge-delta <?= $dc ?>"><?= ($d>=0?'+':'').$d ?></span></td>
              </tr>
            </tbody>
          </table>
        </div>

        <div class="small muted">Hinweis: Beim Speichern wird ausschließlich <strong>IST</strong> (anzahlist) aktualisiert. <strong>SOLL</strong> (anzahl) bleibt unverändert. Zusätzlich wird <strong>lscan</strong> auf 1 gesetzt.</div>

      <?php
      // ===== Ergebnisliste =====
      elseif ($q !== ''):
        if (empty($list)): ?>
          <div class="alert alert-warning mb-0">Kein Gerät zum Barcode „<?= h($q) ?>“ gefunden.</div>
        <?php else: ?>
          <div class="d-flex align-items-center justify-content-between mb-2">
            <h2 class="h6 mb-0"><?= h($headline) ?> (<?= count($list) ?>)</h2>
            <div class="small muted">Einheit: <?= $einheit ? h($einheit) : '—' ?></div>
          </div>
          <div class="table-responsive">
            <table class="table align-middle">
              <thead class="table-light">
                <tr>
                  <th>Bezeichnung</th>
                  <th>Barcode</th>
                  <th>GR</th>
                  <th>Fahrzeug</th>
                  <th>Standort</th>
                  <th class="text-end">SOLL</th>
                  <th class="text-end">IST</th>
                  <th class="text-end">Δ</th>
                  <th class="text-end" style="width:120px;">Aktion</th>
                </tr>
              </thead>
              <tbody>
                <?php foreach ($list as $r):
                  $sol = toInt($r['anzahl']); $ist = toInt($r['anzahlist']); $d=$ist-$sol; $dc=deltaClass($d);
                ?>
                <tr>
                  <td><strong><?= h($r['bezeichnung']) ?></strong><?= (mb_strtolower(trim($r['status']))==='ausgesondert') ? ' <span class="badge bg-danger-subtle text-danger border">ausgesondert</span>' : '' ?></td>
                  <td class="text-nowrap"><?= h($r['barcode']) ?></td>
                  <td><?= h($r['gr']) ?></td>
                  <td><?= h(fahrzeug_label($r)) ?></td>
                  <td><?= h($r['standort']) ?></td>
                  <td class="text-end"><?= $sol ?></td>
                  <td class="text-end"><?= $ist ?></td>
                  <td class="text-end"><span class="badge badge-delta <?= $dc ?>"><?= ($d>=0?'+':'').$d ?></span></td>
                  <td class="text-end">
                    <a class="btn btn-sm btn-primary" href="?id=<?= (int)$r['id'] ?>&q=<?= urlencode($q) ?>">Öffnen</a>
                  </td>
                </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        <?php endif; ?>

      <?php else: ?>
        <div class="alert alert-info mb-0">Tipp: Mit Handscanner fokussiert das Feld automatisch — Enter startet die Suche.</div>
      <?php endif; ?>
    </div>
  </div>
</div>

<script>
(function(){
  // Nur in der Detailansicht vorhanden
  const istInput   = document.getElementById('istInput');
  const badge      = document.getElementById('deltaBadge');
  const cellSoll   = document.getElementById('cellSoll');
  const btnInc     = document.getElementById('btnInc');
  const btnDec     = document.getElementById('btnDec');
  const btnReset   = document.getElementById('btnReset');
  if(!istInput || !badge || !cellSoll) return;

  const getSoll = () => parseInt(cellSoll.textContent.replace(/\D+/g,''),10) || 0;
  const setBadge = (d) => {
    badge.textContent = (d>=0?'+':'') + d;
    badge.classList.remove('pos','neg','zero');
    badge.classList.add(d===0?'zero':(d>0?'pos':'neg'));
  };
  const recalc = () => { const d = (parseInt(istInput.value||'0',10)||0) - getSoll(); setBadge(d); };

  if(btnInc)  btnInc.addEventListener('click', ()=>{ istInput.value = Math.max(0,(parseInt(istInput.value||'0',10)||0)+1); recalc(); });
  if(btnDec)  btnDec.addEventListener('click', ()=>{ istInput.value = Math.max(0,(parseInt(istInput.value||'0',10)||0)-1); recalc(); });
  if(btnReset)btnReset.addEventListener('click',()=>{ istInput.value = getSoll(); recalc(); });
  istInput.addEventListener('input', recalc);
  recalc();
})();
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
